/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *	sysdep.c  --  sysdep API implementation.
 *
 *	Author: Sasha K (sashak@smlink.com)
 *
 *
 */

/*****************************************************************************/

#include <linux/version.h>
#include <linux/module.h>

#include <linux/slab.h>
#include <linux/time.h>
#include <linux/sched.h>
#include <linux/tqueue.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/interrupt.h>
#include <asm/system.h>

#include <linux_compat.h>

/* memory allocations */
void *sysdep_malloc (unsigned long int size)
{
	return kmalloc(size, in_interrupt() ? GFP_ATOMIC : GFP_KERNEL);
}

void sysdep_free (void *mem)
{
	kfree(mem);
}

/* sleep */
void sysdep_sleep(long ms)
{
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_timeout(ms/(1000/HZ));
}

/* getting times */
unsigned long sysdep_get_utime(void)
{
    struct timeval t;
    do_gettimeofday (&t);
    return t.tv_sec * 1000000 + t.tv_usec ;
}

/* cli/sti handling */
void sysdep_save_flags   (unsigned long *flags) {save_flags(*flags); }
void sysdep_restore_flags(unsigned long *flags) {restore_flags(*flags);}
void sysdep_cli(void) {cli();}


/* tasks handling */

void *sysdep_init_task(void (*routine)(void *), void *data)
{
	struct tq_struct *task = sysdep_malloc(sizeof(*task));
	if (!task)
		return NULL;
	memset(task,0,sizeof(*task));
	task->routine = routine;
	task->data    = data;
	return task;
}

void  sysdep_schedule_task(void *task)
{
	if (task)
		schedule_task((struct tq_struct *)task);
}

void  sysdep_cancel_task(void *task)
{
	return;
}

void  sysdep_free_task (void *task)
{
	if (task)
		sysdep_free(task);
}

void sysdep_schedule_event(void *event, void *data)
{
	struct tq_struct *task = event;
	if (task) {
		task->data = data;
		schedule_task(task);
	}
}

/* timers */
void *sysdep_init_timer(void (*function)(unsigned long), void *data) {
    struct timer_list *timer = sysdep_malloc(sizeof(*timer));
    if (!timer)
	return NULL;
    init_timer(timer);  
    timer->data     = (unsigned long)data    ;
    timer->function = function;
    return timer;
}

int  sysdep_add_timer(void *t, unsigned long delay /* ms */) 
{
    struct timer_list *timer = (struct timer_list *)t;
    if (!timer)
	return -1;
    timer->expires  = jiffies + delay*HZ/1000;
    add_timer(timer);
    return 0;
}

void sysdep_del_timer(void* timer)
{
    if (timer)
	del_timer_sync(timer);
}

void sysdep_free_timer(void * timer)
{
	if (timer_pending (timer))
		del_timer_sync(timer);
	sysdep_free(timer);
}

/* export symbols */
EXPORT_SYMBOL_NOVERS(sysdep_malloc);
EXPORT_SYMBOL_NOVERS(sysdep_free);
EXPORT_SYMBOL_NOVERS(sysdep_get_utime);
